/*
   +----------------------------------------------------------------------+
   | Copyright (c) The PHP Group                                          |
   +----------------------------------------------------------------------+
   | This source file is subject to version 3.01 of the PHP license,      |
   | that is bundled with this package in the file LICENSE, and is        |
   | available through the world-wide-web at the following url:           |
   | https://www.php.net/license/3_01.txt                                 |
   | If you did not receive a copy of the PHP license and are unable to   |
   | obtain it through the world-wide-web, please send a note to          |
   | license@php.net so we can mail you a copy immediately.               |
   +----------------------------------------------------------------------+
   | Authors:                                                             |
   | Wez Furlong (wez@thebrainroom.com)                                   |
   | Sara Golemon (pollita@php.net)                                       |
   +----------------------------------------------------------------------+
*/

#include "php.h"
#include "php_globals.h"
#include "ext/standard/basic_functions.h"
#include "ext/standard/file.h"
#include "ext/standard/user_filters_arginfo.h"

#define PHP_STREAM_BRIGADE_RES_NAME	"userfilter.bucket brigade"
#define PHP_STREAM_BUCKET_RES_NAME "userfilter.bucket"

struct php_user_filter_data {
	zend_class_entry *ce;
	/* variable length; this *must* be last in the structure */
	zend_string *classname;
};

/* to provide context for calling into the next filter from user-space */
static int le_bucket_brigade;
static int le_bucket;

/* define the base filter class */

PHP_METHOD(php_user_filter, filter)
{
	zval *in, *out, *consumed;
	bool closing;
	if (zend_parse_parameters(ZEND_NUM_ARGS(), "rrzb", &in, &out, &consumed, &closing) == FAILURE) {
		RETURN_THROWS();
	}

	RETURN_LONG(PSFS_ERR_FATAL);
}

PHP_METHOD(php_user_filter, onCreate)
{
	ZEND_PARSE_PARAMETERS_NONE();

	RETURN_TRUE;
}

PHP_METHOD(php_user_filter, onClose)
{
	ZEND_PARSE_PARAMETERS_NONE();
}

static zend_class_entry *user_filter_class_entry;
static zend_class_entry *stream_bucket_class_entry;

static ZEND_RSRC_DTOR_FUNC(php_bucket_dtor)
{
	php_stream_bucket *bucket = (php_stream_bucket *)res->ptr;
	if (bucket) {
		php_stream_bucket_delref(bucket);
		bucket = NULL;
	}
}

PHP_MINIT_FUNCTION(user_filters)
{
	/* init the filter class ancestor */
	user_filter_class_entry = register_class_php_user_filter();
	stream_bucket_class_entry = register_class_StreamBucket();

	/* Filters will dispose of their brigades */
	le_bucket_brigade = zend_register_list_destructors_ex(NULL, NULL, PHP_STREAM_BRIGADE_RES_NAME, module_number);
	/* Brigades will dispose of their buckets */
	le_bucket = zend_register_list_destructors_ex(php_bucket_dtor, NULL, PHP_STREAM_BUCKET_RES_NAME, module_number);

	if (le_bucket_brigade == FAILURE) {
		return FAILURE;
	}

	register_user_filters_symbols(module_number);

	return SUCCESS;
}

PHP_RSHUTDOWN_FUNCTION(user_filters)
{
	if (BG(user_filter_map)) {
		zend_hash_destroy(BG(user_filter_map));
		efree(BG(user_filter_map));
		BG(user_filter_map) = NULL;
	}

	return SUCCESS;
}

static void userfilter_dtor(php_stream_filter *thisfilter)
{
	zval *obj = &thisfilter->abstract;
	zval retval;

	if (Z_ISUNDEF_P(obj)) {
		/* If there's no object associated then there's nothing to dispose of */
		return;
	}

	zend_string *func_name = ZSTR_INIT_LITERAL("onclose", 0);
	zend_call_method_if_exists(Z_OBJ_P(obj), func_name, &retval, 0, NULL);
	zend_string_release(func_name);

	zval_ptr_dtor(&retval);

	/* kill the object */
	zval_ptr_dtor(obj);
}

static php_stream_filter_status_t userfilter_filter(
			php_stream *stream,
			php_stream_filter *thisfilter,
			php_stream_bucket_brigade *buckets_in,
			php_stream_bucket_brigade *buckets_out,
			size_t *bytes_consumed,
			int flags
			)
{
	int ret = PSFS_ERR_FATAL;
	zval *obj = &thisfilter->abstract;
	zval func_name;
	zval retval;
	zval args[4];
	int call_result;

	/* the userfilter object probably doesn't exist anymore */
	if (CG(unclean_shutdown)) {
		return ret;
	}

	/* Make sure the stream is not closed while the filter callback executes. */
	uint32_t orig_no_fclose = stream->flags & PHP_STREAM_FLAG_NO_FCLOSE;
	stream->flags |= PHP_STREAM_FLAG_NO_FCLOSE;

	zval *stream_prop = zend_hash_str_find_ind(Z_OBJPROP_P(obj), "stream", sizeof("stream")-1);
	if (stream_prop) {
		/* Give the userfilter class a hook back to the stream */
		zval_ptr_dtor(stream_prop);
		php_stream_to_zval(stream, stream_prop);
		Z_ADDREF_P(stream_prop);
	}

	ZVAL_STRINGL(&func_name, "filter", sizeof("filter")-1);

	/* Setup calling arguments */
	ZVAL_RES(&args[0], zend_register_resource(buckets_in, le_bucket_brigade));
	ZVAL_RES(&args[1], zend_register_resource(buckets_out, le_bucket_brigade));

	if (bytes_consumed) {
		ZVAL_LONG(&args[2], *bytes_consumed);
	} else {
		ZVAL_NULL(&args[2]);
	}
	ZVAL_MAKE_REF(&args[2]);

	ZVAL_BOOL(&args[3], flags & PSFS_FLAG_FLUSH_CLOSE);

	call_result = call_user_function(NULL,
			obj,
			&func_name,
			&retval,
			4, args);

	zval_ptr_dtor(&func_name);

	if (call_result == SUCCESS && Z_TYPE(retval) != IS_UNDEF) {
		convert_to_long(&retval);
		ret = (int)Z_LVAL(retval);
	} else if (call_result == FAILURE) {
		php_error_docref(NULL, E_WARNING, "Failed to call filter function");
	}

	if (bytes_consumed) {
		*bytes_consumed = zval_get_long(&args[2]);
	}

	if (buckets_in->head) {
		php_error_docref(NULL, E_WARNING, "Unprocessed filter buckets remaining on input brigade");
	}

	/* filter resources are cleaned up by the stream destructor,
	 * keeping a reference to the stream resource here would prevent it
	 * from being destroyed properly */
	if (stream_prop) {
		convert_to_null(stream_prop);
	}

	zval_ptr_dtor(&args[3]);
	zval_ptr_dtor(&args[2]);
	zval_ptr_dtor(&args[1]);
	zval_ptr_dtor(&args[0]);

	stream->flags &= ~PHP_STREAM_FLAG_NO_FCLOSE;
	stream->flags |= orig_no_fclose;

	return ret;
}

static const php_stream_filter_ops userfilter_ops = {
	userfilter_filter,
	userfilter_dtor,
	"user-filter"
};

static php_stream_filter *user_filter_factory_create(const char *filtername,
		zval *filterparams, uint8_t persistent)
{
	struct php_user_filter_data *fdat = NULL;
	php_stream_filter *filter;
	zval obj;
	zval retval;
	size_t len;

	/* some sanity checks */
	if (persistent) {
		php_error_docref(NULL, E_WARNING,
				"Cannot use a user-space filter with a persistent stream");
		return NULL;
	}

	len = strlen(filtername);

	/* determine the classname/class entry */
	if (NULL == (fdat = zend_hash_str_find_ptr(BG(user_filter_map), (char*)filtername, len))) {
		char *period;

		/* Userspace Filters using ambiguous wildcards could cause problems.
           i.e.: myfilter.foo.bar will always call into myfilter.foo.*
                 never seeing myfilter.*
           TODO: Allow failed userfilter creations to continue
                 scanning through the list */
		if ((period = strrchr(filtername, '.'))) {
			char *wildcard = safe_emalloc(len, 1, 3);

			/* Search for wildcard matches instead */
			memcpy(wildcard, filtername, len + 1); /* copy \0 */
			period = wildcard + (period - filtername);
			while (period) {
				ZEND_ASSERT(period[0] == '.');
				period[1] = '*';
				period[2] = '\0';
				if (NULL != (fdat = zend_hash_str_find_ptr(BG(user_filter_map), wildcard, strlen(wildcard)))) {
					period = NULL;
				} else {
					*period = '\0';
					period = strrchr(wildcard, '.');
				}
			}
			efree(wildcard);
		}
		ZEND_ASSERT(fdat);
	}

	/* bind the classname to the actual class */
	if (fdat->ce == NULL) {
		if (NULL == (fdat->ce = zend_lookup_class(fdat->classname))) {
			php_error_docref(NULL, E_WARNING,
					"User-filter \"%s\" requires class \"%s\", but that class is not defined",
					filtername, ZSTR_VAL(fdat->classname));
			return NULL;
		}
	}

	/* create the object */
	if (object_init_ex(&obj, fdat->ce) == FAILURE) {
		return NULL;
	}

	filter = php_stream_filter_alloc(&userfilter_ops, NULL, 0);

	/* filtername */
	add_property_string(&obj, "filtername", (char*)filtername);

	/* and the parameters, if any */
	if (filterparams) {
		add_property_zval(&obj, "params", filterparams);
	} else {
		add_property_null(&obj, "params");
	}

	/* invoke the constructor */
	zend_string *func_name = ZSTR_INIT_LITERAL("oncreate", 0);
	zend_call_method_if_exists(Z_OBJ(obj), func_name, &retval, 0, NULL);
	zend_string_release(func_name);

	if (Z_TYPE(retval) != IS_UNDEF) {
		if (Z_TYPE(retval) == IS_FALSE) {
			/* User reported filter creation error "return false;" */
			zval_ptr_dtor(&retval);

			/* Kill the filter (safely) */
			ZVAL_UNDEF(&filter->abstract);
			php_stream_filter_free(filter);

			/* Kill the object */
			zval_ptr_dtor(&obj);

			/* Report failure to filter_alloc */
			return NULL;
		}
		zval_ptr_dtor(&retval);
	}

	ZVAL_OBJ(&filter->abstract, Z_OBJ(obj));

	return filter;
}

static const php_stream_filter_factory user_filter_factory = {
	user_filter_factory_create
};

static void filter_item_dtor(zval *zv)
{
	struct php_user_filter_data *fdat = Z_PTR_P(zv);
	zend_string_release_ex(fdat->classname, 0);
	efree(fdat);
}

/* {{{ Return a bucket object from the brigade for operating on */
PHP_FUNCTION(stream_bucket_make_writeable)
{
	zval *zbrigade, zbucket;
	php_stream_bucket_brigade *brigade;
	php_stream_bucket *bucket;

	ZEND_PARSE_PARAMETERS_START(1, 1)
		Z_PARAM_RESOURCE(zbrigade)
	ZEND_PARSE_PARAMETERS_END();

	if ((brigade = (php_stream_bucket_brigade*)zend_fetch_resource(
					Z_RES_P(zbrigade), PHP_STREAM_BRIGADE_RES_NAME, le_bucket_brigade)) == NULL) {
		RETURN_THROWS();
	}

	if (brigade->head && (bucket = php_stream_bucket_make_writeable(brigade->head))) {
		ZVAL_RES(&zbucket, zend_register_resource(bucket, le_bucket));
		object_init_ex(return_value, stream_bucket_class_entry);
		zend_update_property(Z_OBJCE_P(return_value), Z_OBJ_P(return_value), ZEND_STRL("bucket"), &zbucket);
		/* zend_update_property increments the refcount which is unwanted here */
		Z_DELREF(zbucket);
		zend_update_property_stringl(Z_OBJCE_P(return_value), Z_OBJ_P(return_value), ZEND_STRL("data"), bucket->buf, bucket->buflen);
		zend_update_property_long(Z_OBJCE_P(return_value), Z_OBJ_P(return_value), ZEND_STRL("datalen"), bucket->buflen);
		zend_update_property_long(Z_OBJCE_P(return_value), Z_OBJ_P(return_value), ZEND_STRL("dataLength"), bucket->buflen);
	} else {
		ZVAL_NULL(return_value);
	}
}
/* }}} */

/* {{{ php_stream_bucket_attach */
static void php_stream_bucket_attach(int append, INTERNAL_FUNCTION_PARAMETERS)
{
	zval *zbrigade, *zobject;
	zval *pzbucket, *pzdata, rv;
	php_stream_bucket_brigade *brigade;
	php_stream_bucket *bucket;

	ZEND_PARSE_PARAMETERS_START(2, 2)
		Z_PARAM_RESOURCE(zbrigade)
		Z_PARAM_OBJECT_OF_CLASS(zobject, stream_bucket_class_entry)
	ZEND_PARSE_PARAMETERS_END();

	if ((brigade = (php_stream_bucket_brigade*)zend_fetch_resource(
		Z_RES_P(zbrigade), PHP_STREAM_BRIGADE_RES_NAME, le_bucket_brigade)) == NULL) {
		RETURN_THROWS();
	}

	if (NULL == (pzbucket = zend_read_property(NULL, Z_OBJ_P(zobject), "bucket", sizeof("bucket")-1, false, &rv))) {
		zend_argument_value_error(2, "must be an object that has a \"bucket\" property");
		RETURN_THROWS();
	}
	ZVAL_DEREF(pzbucket);

	if ((bucket = (php_stream_bucket *)zend_fetch_resource_ex(pzbucket, PHP_STREAM_BUCKET_RES_NAME, le_bucket)) == NULL) {
		RETURN_THROWS();
	}

	if (NULL != (pzdata = zend_read_property(NULL, Z_OBJ_P(zobject), "data", sizeof("data")-1, false, &rv))) {
		ZVAL_DEREF(pzdata);
		if (!bucket->own_buf) {
			bucket = php_stream_bucket_make_writeable(bucket);
		}
		if (bucket->buflen != Z_STRLEN_P(pzdata)) {
			bucket->buf = perealloc(bucket->buf, MAX(Z_STRLEN_P(pzdata), 1), bucket->is_persistent);
			bucket->buflen = Z_STRLEN_P(pzdata);
		}
		memcpy(bucket->buf, Z_STRVAL_P(pzdata), bucket->buflen);
	}

	/* If the bucket is already on a brigade we have to unlink it first to keep the
	 * linked list consistent. Furthermore, we can transfer the refcount in that case. */
	if (bucket->brigade) {
		php_stream_bucket_unlink(bucket);
	} else {
		bucket->refcount++;
	}

	if (append) {
		php_stream_bucket_append(brigade, bucket);
	} else {
		php_stream_bucket_prepend(brigade, bucket);
	}
}
/* }}} */

/* {{{ Prepend bucket to brigade */
PHP_FUNCTION(stream_bucket_prepend)
{
	php_stream_bucket_attach(0, INTERNAL_FUNCTION_PARAM_PASSTHRU);
}
/* }}} */

/* {{{ Append bucket to brigade */
PHP_FUNCTION(stream_bucket_append)
{
	php_stream_bucket_attach(1, INTERNAL_FUNCTION_PARAM_PASSTHRU);
}
/* }}} */

/* {{{ Create a new bucket for use on the current stream */
PHP_FUNCTION(stream_bucket_new)
{
	zval zbucket;
	php_stream *stream;
	char *buffer;
	char *pbuffer;
	size_t buffer_len;
	php_stream_bucket *bucket;

	ZEND_PARSE_PARAMETERS_START(2, 2)
		PHP_Z_PARAM_STREAM(stream)
		Z_PARAM_STRING(buffer, buffer_len)
	ZEND_PARSE_PARAMETERS_END();

	pbuffer = pemalloc(buffer_len, php_stream_is_persistent(stream));

	memcpy(pbuffer, buffer, buffer_len);

	bucket = php_stream_bucket_new(stream, pbuffer, buffer_len, 1, php_stream_is_persistent(stream));

	ZVAL_RES(&zbucket, zend_register_resource(bucket, le_bucket));
	object_init_ex(return_value, stream_bucket_class_entry);
	zend_update_property(Z_OBJCE_P(return_value), Z_OBJ_P(return_value), ZEND_STRL("bucket"), &zbucket);
	/* zend_update_property increments the refcount which is unwanted here */
	Z_DELREF(zbucket);
	zend_update_property_stringl(Z_OBJCE_P(return_value), Z_OBJ_P(return_value), ZEND_STRL("data"), bucket->buf, bucket->buflen);
	zend_update_property_long(Z_OBJCE_P(return_value), Z_OBJ_P(return_value), ZEND_STRL("datalen"), bucket->buflen);
	zend_update_property_long(Z_OBJCE_P(return_value), Z_OBJ_P(return_value), ZEND_STRL("dataLength"), bucket->buflen);
}
/* }}} */

/* {{{ Returns a list of registered filters */
PHP_FUNCTION(stream_get_filters)
{
	zend_string *filter_name;
	HashTable *filters_hash;

	ZEND_PARSE_PARAMETERS_NONE();


	filters_hash = php_get_stream_filters_hash();

	if (filters_hash && !HT_IS_PACKED(filters_hash)) {
		array_init(return_value);
		zend_hash_real_init_packed(Z_ARRVAL_P(return_value));
		ZEND_HASH_MAP_FOREACH_STR_KEY(filters_hash, filter_name) {
			if (filter_name) {
				add_next_index_str(return_value, zend_string_copy(filter_name));
			}
		} ZEND_HASH_FOREACH_END();
	} else {
		RETURN_EMPTY_ARRAY();
	}
	/* It's okay to return an empty array if no filters are registered */
}
/* }}} */

/* {{{ Registers a custom filter handler class */
PHP_FUNCTION(stream_filter_register)
{
	zend_string *filtername, *classname;
	struct php_user_filter_data *fdat;

	ZEND_PARSE_PARAMETERS_START(2, 2)
		Z_PARAM_STR(filtername)
		Z_PARAM_STR(classname)
	ZEND_PARSE_PARAMETERS_END();

	if (!ZSTR_LEN(filtername)) {
		zend_argument_value_error(1, "must be a non-empty string");
		RETURN_THROWS();
	}

	if (!ZSTR_LEN(classname)) {
		zend_argument_value_error(2, "must be a non-empty string");
		RETURN_THROWS();
	}

	if (!BG(user_filter_map)) {
		BG(user_filter_map) = (HashTable*) emalloc(sizeof(HashTable));
		zend_hash_init(BG(user_filter_map), 8, NULL, (dtor_func_t) filter_item_dtor, 0);
	}

	fdat = ecalloc(1, sizeof(struct php_user_filter_data));
	fdat->classname = zend_string_copy(classname);

	if (zend_hash_add_ptr(BG(user_filter_map), filtername, fdat) != NULL) {
		if (php_stream_filter_register_factory_volatile(filtername, &user_filter_factory) == SUCCESS) {
			RETURN_TRUE;
		}

		zend_hash_del(BG(user_filter_map), filtername);
	} else {
		zend_string_release_ex(classname, 0);
		efree(fdat);
	}

	RETURN_FALSE;
}
/* }}} */
